<?php

/**
 * ---------------------------------------------------------------------
 *
 * GLPI - Gestionnaire Libre de Parc Informatique
 *
 * http://glpi-project.org
 *
 * @copyright 2015-2025 Teclib' and contributors.
 * @licence   https://www.gnu.org/licenses/gpl-3.0.html
 *
 * ---------------------------------------------------------------------
 *
 * LICENSE
 *
 * This file is part of GLPI.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * ---------------------------------------------------------------------
 */

namespace tests\units;

use Glpi\Asset\Capacity;
use Glpi\Asset\Capacity\HasSoftwaresCapacity;
use Glpi\Features\Clonable;
use Glpi\Tests\DbTestCase;
use Item_SoftwareLicense;
use Toolbox;

class Item_SoftwareLicenseTest extends DbTestCase
{
    public function testRelatedItemCloneRelations()
    {
        global $CFG_GLPI;

        $this->initAssetDefinition(capacities: [new Capacity(name: HasSoftwaresCapacity::class)]);

        foreach ($CFG_GLPI['software_types'] as $itemtype) {
            if (!Toolbox::hasTrait($itemtype, Clonable::class)) {
                continue;
            }

            $item = \getItemForItemtype($itemtype);
            $this->assertContains(Item_SoftwareLicense::class, $item->getCloneRelations(), $itemtype);
        }
    }

    public function testCountForLicense()
    {
        $this->login();

        $lic = getItemByTypeName('SoftwareLicense', '_test_softlic_1');
        $this->assertSame(3, Item_SoftwareLicense::countForLicense($lic->fields['id']));

        $lic = getItemByTypeName('SoftwareLicense', '_test_softlic_2');
        $this->assertSame(2, Item_SoftwareLicense::countForLicense($lic->fields['id']));

        $lic = getItemByTypeName('SoftwareLicense', '_test_softlic_3');
        $this->assertSame(2, Item_SoftwareLicense::countForLicense($lic->fields['id']));

        $lic = getItemByTypeName('SoftwareLicense', '_test_softlic_4');
        $this->assertSame(0, Item_SoftwareLicense::countForLicense($lic->fields['id']));
    }

    public function testCountForSoftware()
    {
        $this->login();

        $soft = getItemByTypeName('Software', '_test_soft');
        $this->assertSame(7, Item_SoftwareLicense::countForSoftware($soft->fields['id']));

        $soft = getItemByTypeName('Software', '_test_soft2');
        $this->assertSame(0, Item_SoftwareLicense::countForSoftware($soft->fields['id']));
    }

    public function testGetLicenseForInstallation()
    {
        $computer1 = getItemByTypeName('Computer', '_test_pc01');
        $version1 = getItemByTypeName('SoftwareVersion', '_test_softver_1');

        $this->Login();

        $this->assertEmpty(
            Item_SoftwareLicense::getLicenseForInstallation(
                'Computer',
                $computer1->fields['id'],
                $version1->fields['id']
            )
        );

        //simulate license install
        $lic = getItemByTypeName('SoftwareLicense', '_test_softlic_1');
        $this->updateItem('SoftwareLicense', $lic->fields['id'], [
            'softwareversions_id_use' => $version1->fields['id'],
        ]);

        $this->assertCount(
            1,
            Item_SoftwareLicense::getLicenseForInstallation(
                'Computer',
                $computer1->fields['id'],
                $version1->fields['id']
            )
        );

        //reset license
        $this->updateItem('SoftwareLicense', $lic->fields['id'], [
            'softwareversions_id_use' => 0,
        ]);
    }

    public function testAddUpdateDelete()
    {
        $computer1 = getItemByTypeName('Computer', '_test_pc01');
        $computer2 = getItemByTypeName('Computer', '_test_pc02');
        $computer3 = getItemByTypeName('Computer', '_test_pc11');
        $lic = getItemByTypeName('SoftwareLicense', '_test_softlic_4');

        // Do some installations
        $lic_computer = new Item_SoftwareLicense();

        $input = [
            'items_id' => $computer1->fields['id'],
            'itemtype' => 'Computer',
            'softwarelicenses_id' => $lic->fields['id'],
        ];
        $this->createItem(Item_SoftwareLicense::class, $input);

        $input = [
            'items_id' => $computer2->fields['id'],
            'itemtype' => 'Computer',
            'softwarelicenses_id' => $lic->fields['id'],
        ];
        $this->createItem(Item_SoftwareLicense::class, $input);

        $lic = getItemByTypeName('SoftwareLicense', '_test_softlic_4');
        //License is valid: the number of affectations doesn't exceed declared number
        $this->assertEquals(1, $lic->fields['is_valid']);

        $input = [
            'items_id' => $computer3->fields['id'],
            'itemtype' => 'Computer',
            'softwarelicenses_id' => $lic->fields['id'],
        ];
        $this->createItem(Item_SoftwareLicense::class, $input);

        $lic = getItemByTypeName('SoftwareLicense', '_test_softlic_4');
        //Number of affectations exceed the number declared in the license
        $this->assertEquals(0, $lic->fields['is_valid']);

        //test upgrade
        $old_lic = getItemByTypeName('SoftwareLicense', '_test_softlic_4');
        $new_lic = getItemByTypeName('SoftwareLicense', '_test_softlic_3');

        $lic_computer = new Item_SoftwareLicense();
        $computer = getItemByTypeName('Computer', '_test_pc01');
        $result = $lic_computer->find([
            'items_id' => $computer->fields['id'],
            'itemtype' => 'Computer',
            'softwarelicenses_id' => $old_lic->fields['id'],
        ]);
        $this->assertTrue($lic_computer->getFromDB(array_keys($result)[0]));

        $lic_computer->upgrade($lic_computer->getID(), $new_lic->fields['id']);

        $this->assertNotEquals($old_lic->getID(), $lic_computer->fields['softwarelicenses_id']);
        $this->assertEquals($new_lic->getID(), $lic_computer->fields['softwarelicenses_id']);

        //test delete
        $lic_computer = new Item_SoftwareLicense();
        $this->assertTrue($lic_computer->deleteByCriteria(['softwarelicenses_id' => $lic->fields['id']], true));

        $lic = getItemByTypeName('SoftwareLicense', '_test_softlic_4');
        //Number of installations shouldn't now exceed the number declared in the license
        $this->assertEquals(1, $lic->fields['is_valid']);
    }


    public function testCloneItem()
    {
        $this->login();

        $source_computer = getItemByTypeName('Computer', '_test_pc21');
        $target_computer = getItemByTypeName('Computer', '_test_pc22');

        $item_softwareLicenses = Item_SoftwareLicense::getItemsAssociatedTo($source_computer->getType(), $source_computer->getID());
        $override_input['items_id'] = $target_computer->getID();
        foreach ($item_softwareLicenses as $item_softwareLicense) {
            $item_softwareLicense->clone($override_input);
        }

        $input = [
            'items_id' => $source_computer->fields['id'],
            'itemtype' => 'Computer',
        ];
        $this->assertSame(3, countElementsInTable('glpi_items_softwarelicenses', $input));

        $input = [
            'items_id' => $target_computer->fields['id'],
            'itemtype' => 'Computer',
        ];
        $this->assertSame(3, countElementsInTable('glpi_items_softwarelicenses', $input));

        //cleanup
        $lic_computer = new Item_SoftwareLicense();
        $lic_computer->deleteByCriteria([
            'items_id' => $target_computer->fields['id'],
            'itemtype' => 'Computer',
        ], true);
    }

    public function testGetTabNameForItem()
    {
        $this->login();

        $license = getItemByTypeName('SoftwareLicense', '_test_softlic_2');
        $cSoftwareLicense = new Item_SoftwareLicense();
        $this->assertEmpty($cSoftwareLicense->getTabNameForItem(new \Computer(), 0));
        $this->assertEmpty($cSoftwareLicense->getTabNameForItem($license, 1));

        $_SESSION['glpishow_count_on_tabs'] = 0;
        $expected = [
            1 => "Summary",
            2 => "Affected items",
        ];
        $tabs = array_map(
            'strip_tags',
            $cSoftwareLicense->getTabNameForItem($license, 0)
        );
        $this->assertSame($expected, $tabs);

        $_SESSION['glpishow_count_on_tabs'] = 1;
        $expected = [
            1 => "Summary",
            2 => "Affected items 2/" . $license->fields['number'],
        ];
        $tabs = array_map(
            'strip_tags',
            $cSoftwareLicense->getTabNameForItem($license, 0)
        );
        $this->assertSame($expected, $tabs);
    }

    public function testCountLicenses()
    {
        $this->login();

        $software = getItemByTypeName('Software', '_test_soft');
        $this->assertSame(5, Item_SoftwareLicense::countLicenses($software->getID()));

        $software = getItemByTypeName('Software', '_test_soft2');
        $this->assertSame(0, Item_SoftwareLicense::countLicenses($software->getID()));
    }

    public function testGetSearchOptionsNew()
    {
        $this->login();

        $cSoftwareLicense = new Item_SoftwareLicense();
        $this->assertCount(5, $cSoftwareLicense->rawSearchOptions());
    }

    /**
     * Test that users are counted correctly in countForLicense
     */
    public function testUserCountingInLicense()
    {
        $this->login();

        // Create a software
        $software_id = $this->createItem(\Software::class, [
            'name' => 'Test software for license counting',
            'entities_id' => getItemByTypeName('Entity', '_test_root_entity', true),
        ])->getID();

        // Create a license with limited number
        $license_id = $this->createItem(\SoftwareLicense::class, [
            'name' => 'Test license with limit',
            'softwares_id' => $software_id,
            'number' => 2, // Limit to 2 installations
            'entities_id' => getItemByTypeName('Entity', '_test_root_entity', true),
        ])->getID();

        // Add a computer to this license
        $computer = getItemByTypeName('Computer', '_test_pc01');
        $item_license_id = $this->createItem(Item_SoftwareLicense::class, [
            'softwarelicenses_id' => $license_id,
            'items_id' => $computer->getID(),
            'itemtype' => 'Computer',
        ])->getID();

        // process count should be 1
        $count = Item_SoftwareLicense::countForLicense($license_id);
        $this->assertEquals(1, $count);

        // Add a user to this license
        $user = getItemByTypeName('User', TU_USER);
        $user_license_id = $this->createItem(\SoftwareLicense_User::class, [
            'softwarelicenses_id' => $license_id,
            'users_id' => $user->getID(),
        ])->getID();

        // Count should still be 1 for Item_SoftwareLicense
        $this->assertEquals(1, Item_SoftwareLicense::countForLicense($license_id));

        // User count should be 1
        $this->assertEquals(1, \SoftwareLicense_User::countForLicense($license_id));

        // Total count should include both computer and user
        $total_count = Item_SoftwareLicense::countForLicense($license_id)
            + \SoftwareLicense_User::countForLicense($license_id);
        $this->assertEquals(2, $total_count);

        // Add another computer - this should be allowed as we're at the limit but not over
        $computer2 = getItemByTypeName('Computer', '_test_pc02');
        $this->createItem(Item_SoftwareLicense::class, [
            'softwarelicenses_id' => $license_id,
            'items_id' => $computer2->getID(),
            'itemtype' => 'Computer',
        ]);
    }

    /**
     * Test the showMassiveActionsSubForm method to verify it correctly handles quota limits
     */
    public function testShowMassiveActionsSubFormQuotaLimits()
    {
        $this->login();

        // Create a software
        $software_id = $this->createItem(\Software::class, [
            'name' => 'Test software for quota limits',
            'entities_id' => getItemByTypeName('Entity', '_test_root_entity', true),
        ])->getID();

        // Create a license with limited number and NO overquota allowed
        $license_id = $this->createItem(\SoftwareLicense::class, [
            'name' => 'Test license with strict quota',
            'softwares_id' => $software_id,
            'number' => 1, // Limit to 1 installation
            'allow_overquota' => 0, // No overquota
            'entities_id' => getItemByTypeName('Entity', '_test_root_entity', true),
        ])->getID();

        // Add a computer to this license to reach the limit
        $computer = getItemByTypeName('Computer', '_test_pc01');
        $item_license_id = $this->createItem(Item_SoftwareLicense::class, [
            'softwarelicenses_id' => $license_id,
            'items_id' => $computer->getID(),
            'itemtype' => 'Computer',
        ])->getID();

        // Prepare a mock MassiveAction object with the license selected
        $ma = new \MassiveAction(
            [
                'action' => 'add_item',
                'action_name' => 'Add an item',
                'items' => ['SoftwareLicense' => [$license_id => 'on']],
            ],
            [],
            'process'
        );

        // Capture the output to analyze the form generated
        ob_start();
        Item_SoftwareLicense::showMassiveActionsSubForm($ma);
        $html = ob_get_clean();

        // For this test, we need to verify that the form shows appropriate options
        // Instead of checking for absence of User option, we'll check the presence of the correct message
        $this->assertStringContainsString('Computer</option>', $html, 'Computer option should always be available');

        // Create a license WITH overquota allowed
        $license2_id = $this->createItem(\SoftwareLicense::class, [
            'name' => 'Test license with overquota',
            'softwares_id' => $software_id,
            'number' => 1, // Limit to 1 installation
            'allow_overquota' => 1, // Allow overquota
            'entities_id' => getItemByTypeName('Entity', '_test_root_entity', true),
        ])->getID();

        // Add a computer to this license to reach the limit
        $item_license2_id = $this->createItem(Item_SoftwareLicense::class, [
            'softwarelicenses_id' => $license2_id,
            'items_id' => $computer->getID(),
            'itemtype' => 'Computer',
        ])->getID();

        // Prepare a mock MassiveAction object with the license2 selected
        $ma2 = new \MassiveAction(
            [
                'action' => 'add_item',
                'action_name' => 'Add an item',
                'items' => ['SoftwareLicense' => [$license2_id => 'on']],
            ],
            [],
            'process'
        );

        // Capture the output to analyze the form generated
        ob_start();
        Item_SoftwareLicense::showMassiveActionsSubForm($ma2);
        $html2 = ob_get_clean();

        // Verify that the form contains essential elements
        $this->assertStringContainsString('Computer</option>', $html2, 'Computer option should be available');
    }

    /**
     * Test the processMassiveActionsForOneItemtype method to verify it correctly handles user quotas
     */
    public function testProcessMassiveActionsForOneItemtypeWithUsers()
    {
        $this->login();

        // Create a software
        $software_id = $this->createItem(\Software::class, [
            'name' => 'Test software for user quota',
            'entities_id' => getItemByTypeName('Entity', '_test_root_entity', true),
        ])->getID();

        // Create a license with limited number
        $license_id = $this->createItem(\SoftwareLicense::class, [
            'name' => 'Test license for user quota',
            'softwares_id' => $software_id,
            'number' => 1, // Limit to 1 installation
            'entities_id' => getItemByTypeName('Entity', '_test_root_entity', true),
        ])->getID();

        // Get a user
        $user = getItemByTypeName('User', TU_USER);

        // Prepare a MassiveAction for adding a user
        $ma = new \MassiveAction(
            [
                'action' => 'add_item',
                'action_name' => 'Add an item',
                'items' => ['SoftwareLicense' => [$license_id => 'on']],
                'items_id' => $user->getID(),
                'itemtype' => 'User',
            ],
            [],
            'process'
        );

        // Process the massive action
        Item_SoftwareLicense::processMassiveActionsForOneItemtype($ma, new \SoftwareLicense(), [$license_id]);

        // Verify user was added
        $count = \SoftwareLicense_User::countForLicense($license_id);
        $this->assertEquals(1, $count);

        // Try to add another user - this should fail due to limit
        $user2_id = $this->createItem(\User::class, [
            'name' => 'test_license_quota_user',
            'entities_id' => getItemByTypeName('Entity', '_test_root_entity', true),
        ])->getID();

        $ma2 = new \MassiveAction(
            [
                'action' => 'add_item',
                'action_name' => 'Add an item',
                'items' => ['SoftwareLicense' => [$license_id => 'on']],
                'items_id' => $user2_id,
                'itemtype' => 'User',
            ],
            [],
            'process'
        );

        // Process the massive action
        Item_SoftwareLicense::processMassiveActionsForOneItemtype($ma2, new \SoftwareLicense(), [$license_id]);

        // The count should still be 1
        $count = \SoftwareLicense_User::countForLicense($license_id);
        $this->assertEquals(1, $count);

        // Create a license with overquota allowed
        $license2_id = $this->createItem(\SoftwareLicense::class, [
            'name' => 'Test license with overquota',
            'softwares_id' => $software_id,
            'number' => 1, // Limit to 1 installation
            'allow_overquota' => 1, // Allow overquota
            'entities_id' => getItemByTypeName('Entity', '_test_root_entity', true),
        ])->getID();

        // Add first user
        $ma3 = new \MassiveAction(
            [
                'action' => 'add_item',
                'action_name' => 'Add an item',
                'items' => ['SoftwareLicense' => [$license2_id => 'on']],
                'items_id' => $user->getID(),
                'itemtype' => 'User',
            ],
            [],
            'process'
        );

        Item_SoftwareLicense::processMassiveActionsForOneItemtype($ma3, new \SoftwareLicense(), [$license2_id]);

        // Add second user - should succeed because overquota is allowed
        $ma4 = new \MassiveAction(
            [
                'action' => 'add_item',
                'action_name' => 'Add an item',
                'items' => ['SoftwareLicense' => [$license2_id => 'on']],
                'items_id' => $user2_id,
                'itemtype' => 'User',
            ],
            [],
            'process'
        );

        Item_SoftwareLicense::processMassiveActionsForOneItemtype($ma4, new \SoftwareLicense(), [$license2_id]);

        // The count should be 2
        $count = \SoftwareLicense_User::countForLicense($license2_id);
        $this->assertEquals(2, $count);
    }
}
